/*
 * CO2-Guard
 * 
 * This work is licensed under a Creative Commons Attribution-NonCommercial-ShareAlike 4.0 International License: 
 * CC BY-NC-SA 4.0 https://creativecommons.org/licenses/by-nc-sa/4.0/
 * 
 * Florian Schäffer, https://www.co2guard.de/
 * 
 * CO2-Messung (und Temperatur) 
 * Wemos NodeMCU D1 mini mit ESP8266/ESP12 und Winsen MH-Z19C (https://www.winsen-sensor.com/) CO2-Sensor
 * Die Meßwerte können per WLAN (IoT) an einen Cloud-Server (https://thingspeak.com/) gesendet werden.
 * Es werden zwei Werte gesendet:
 * field1: CO2-Konzentration in ppm
 * field2: Temperatur in °C, ganzzahlig, ungenau
 * 
 * 3 LEDs (Rot, Gelb, Grün und im Taster Blau)
 * 
 * Der Taster wird direkt an den Pin zero point calibration des MH-Z19 angeschlossen. 7 Sekunden drücken nachdem er mind. 20 Minuten
 * an frischer Luft stand.
 * Den kleinen Luefter per PWM zu steuern bringt nichts: Geraeusche lauter
 * Die LED am Taster kann an D4 angeschlossen werden, dann leuchtet sie zusammen mit der blauen LED auf dem D1 Mini
 * 
 * 1.0    28.11.2020
 * 1.1    2.1.2021
 *        - Beep after start immedeatly
 *        - no endless waiting for WiFi. blue LED off if no WiFi
 * 
 * Hinweise:
 * - Wenn kein connect beim übertragen: vom USB-Port abziehen, dann neu übertragen
 * - delay() ist irgendwie ein Problem und laesst den NodeMCU in einer endlos-Boot-Schleife enden. Deshalb unbedingt ESP.wdtDisable();
 * 
 * Wussten Sie?   Glühlampen werden in Russland seit jeher auch Iljitsch-Lämpchen genannt. 
 *                Das erinnert an den Revolutionsführer Wladimir Iljitsch Lenin, der die Elektrifizierung des Landes vorangetrieben hat. 
 * 
 */

#include "config.h"
#include <ESP8266WiFi.h>
#include <ESP8266HTTPClient.h>
#include <Ticker.h>
#include <SoftwareSerial.h>

Ticker timer;
SoftwareSerial co2Serial(RxD, TxD); //   rxPin, txPin       

void ICACHE_RAM_ATTR onTimer (void);
void FlashBlueLED (void);
void PlayAlarm (void);
int32_t ReadCO2(int8_t *);
void WLANConnect (void);

int8_t AlarmStatus = -1;      // -1 = off , 1 = on

void setup() {}   // useless

//-------------------------------------------------------------------------------------------------

/*
  @brief  blaue WLAN-LED blinken lassen
  @param  none
  @return none
*/
void FlashBlueLED (void)
{
  for (uint8_t i=0; i<10; i++)
  {
    digitalWrite(LED_BUILTIN, HIGH);    // blue LED off
    delay (50);
    digitalWrite(LED_BUILTIN, LOW);    // blue LED on
    delay (50);
  }
}

/*
  @brief  ISR fuer Timer: Alarm regelmaessig ausgeben
          move the function to the internal RAM instead of Flash. IMPORTANT!!! (maybe IRAM_ATTR) - bug by ESP
  @param  none
  @return none
*/
void ICACHE_RAM_ATTR onTimer() 
{
  if (AlarmStatus > 0)
    PlayAlarm();
}

/*
  @brief  Laesst den Buzzer ertoenen
  @param  none
  @return none
*/
void PlayAlarm ()
{
#ifdef ALARM_SUPPORT  
  for (uint8_t i = 0; i <= 8; i++)
  {
    digitalWrite(BUZZER, HIGH);   
    digitalWrite(LEDred, LOW);      // alarm => LED is already on
    delayMicroseconds (50000);      // Bug in ESP8266 libraries: do not use "delay" 
    digitalWrite(BUZZER, LOW);  
    digitalWrite(LEDred, HIGH);   
    delayMicroseconds (50000);
  }
#endif
}

/*
  @brief  Fragt den MH-Z19 CO2 Sensor ab
  @param  temp
  @return CO2-Wert in ppm, -1 bei Fehler
          *temp: Temperatur in °C [-40..215]
*/
int32_t ReadCO2(int8_t *temp)                         // Kommunikation mit MH-Z19 CO2 Sensor
{
/*
  * Command 
 * 0xFF CM HH LL TT SS Uh Ul CS
wobei
CM der Befehl ist, der wiederholt wird.
HH / LL ist der CO2-ppm-Wert, High / Low-Teil
TT ist die Temperatur in Grad Celsius plus 40. Wenn zum Beispiel die Temperatur 25 ° C beträgt, dann ist TT = 0x41
SS eine Art Statusbyte , für dieses Byte ist immer nur ein Bit gesetzt!
Uh / Ul ist ein unbekannter Wert, vielleicht im Zusammenhang mit Druck? Nach dem Booten des Sensors beginnt er genau bei 15000 und erreicht dann normalerweise etwa 10500.
Cs ist die Prüfsumme

dies gilz nur fuer die aelteren Modelle
beim Modell C ist SS immer 0, und U1 auch. Uh ist ein Zaehler, der von 0 bis 20 zaehlt.

Aha! Jetzt ist alles klar! Wenn alles in Ordnung ist, ist S gleich 64, und wenn der CO2-Sensor zu würfeln beginnt, fällt er auf 4. 
Mit S können Sie also herausfinden, wie gut sich der Sensor anfühlt und wie genau er ist. 

Preheat time 1 min
Response Time T90 < 120 s

* 
 */
 
  byte cmd[9] = {0xFF, 0x01, 0x86, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79};   // cmd Anforderung data
  char antwort[9];
  char buf[3];

  co2Serial.write(cmd, 9);                        // auch ohne Sensor kein endloss warten
  co2Serial.readBytes(antwort, 9);                // Bsp.: ff 86 03 46 40 00 00 00 f1 

#ifdef DEBUG 
  Serial.print ("MH-Z19: ");
  for (int i=0; i<9; i++)
  {
    sprintf(buf, "%02x", antwort[i]); 
    Serial.print (buf);
    Serial.print (" ");
  }
  Serial.println();
#endif
  
  if (antwort[0] != 0xFF) return -1;
  if (antwort[1] != 0x86) return -1;
  if ((antwort[2] == 0xFF) && (antwort[3] == 0xFF)) return -1;    // 65535 ppm
  *temp = (uint8_t) antwort[4] - 40;        // °C mit Offset 40 
  return (256 * (uint8_t) antwort[2]) + (uint8_t) antwort[3];                         // Antwort des MH-Z19 CO2 Sensors in ppm
}

/*
  @brief  Fragt den MH-Z19 CO2 Sensor ab
  @param  temp, quality
  @return CO2-Wert in ppm, -1 bei Fehler
          *temp: Temperatur in °C [-40..215]
          *quality: Qualitaet Messwert
*/
void WLANConnect (void)
{
  uint8_t i=0;
  
  digitalWrite(LED_BUILTIN, HIGH);     // blue LED off
  WiFi.begin(ssid, password);             // Connect to the network
  #ifdef DEBUG 
    Serial.print("Connecting to "); 
    Serial.print(ssid); Serial.println(" ...");
  #endif

  while ((WiFi.status() != WL_CONNECTED) && (i < 10))  // Wait for the Wi-Fi to connect but max 10 times. endless reconect trys after abort automatically
  {
    delay(1000);
    #ifdef DEBUG 
      Serial.print(++i); Serial.print("...");
    #endif
  }

  if (WiFi.status() == WL_CONNECTED)  // Wi-Fi connected?
  {
    #ifdef DEBUG 
      Serial.print("\nWiFi connected\nIP address: ");  
      Serial.println(WiFi.localIP());         // Send the IP address of the ESP8266 to the computer
    #endif
    
    FlashBlueLED ();
    digitalWrite(LED_BUILTIN, LOW);     // blue LED on
  }
  else
  {
    #ifdef DEBUG 
      Serial.print("\nWiFi _NOT_ connected\n");  
    #endif

    digitalWrite(LED_BUILTIN, HIGH);     // blue LED off
  }
}

/*
  @brief  main()
  @param  none
  @return none
*/
void loop() 
{
  uint8_t i = 0, cnt = 0;
  int32_t co2Wert;
  int8_t temp;
  int16_t httpCode;
#ifdef WLAN_SUPPORT
  String APIurl; 
#endif

  ESP.wdtDisable();     // WICHTIG !!! WICHTIG !!! WICHTIG !!! WICHTIG !!! 

  pinMode(LED_BUILTIN, OUTPUT);       // indicate WiFi
  pinMode(LEDred, OUTPUT);       
  pinMode(LEDylw, OUTPUT);       
  pinMode(LEDgrn, OUTPUT);       
  pinMode(BUZZER, OUTPUT);       
 
  digitalWrite(LED_BUILTIN, HIGH);     // no WiFi. blue LED inverse
  digitalWrite(LEDred, LOW);     
  digitalWrite(LEDylw, LOW);     
  digitalWrite(LEDgrn, LOW);     
  digitalWrite(BUZZER, LOW);     

  // Start-Test
  digitalWrite(BUZZER, HIGH);   
  delay (80);
  digitalWrite(BUZZER, LOW);   

  HTTPClient http;

  Serial.begin(9600);         // Start the Serial communication (MH-Z19 uses 9600 Bd)
  co2Serial.begin(9600);
  
#ifdef WLAN_SUPPORT
  #ifdef DEBUG 
    Serial.setDebugOutput(true);    // enable diagnostic output from WiFi libraries
  #endif
  WLANConnect();
#endif

  for (i = 0; i < 40; i++)        // Preheat-time 60s (1,5s x 40) : flash LEDs
  {
    digitalWrite(LEDred, HIGH);
    digitalWrite(LEDylw, HIGH);
    digitalWrite(LEDgrn, HIGH);
    delay (1000);     
    digitalWrite(LEDred, LOW);
    digitalWrite(LEDylw, LOW);
    digitalWrite(LEDgrn, LOW);
    delay (500);     
  }
  

  // attachInterrupt(digitalPinToInterrupt(SWITCH), KeyPressed, FALLING);      // IRQ fuer Taster

  // Alle 10 s. Alarm pruefen und ggf. ausgeben 
  /* Dividers:
    TIM_DIV1 = 0,   //80MHz (80 ticks/us - 104857.588 us max)
    TIM_DIV16 = 1,  //5MHz (5 ticks/us - 1677721.4 us max)
    TIM_DIV256 = 3  //312.5Khz (1 tick = 3.2us - 26843542.4 us max)
  Reloads:
    TIM_SINGLE  0 //on interrupt routine you need to write a new value to start the timer again
    TIM_LOOP  1 //on interrupt the counter will start with the same value again
  */
  timer1_attachInterrupt(onTimer);      // Add ISR Function for alarm buzzer
  timer1_enable(TIM_DIV256, TIM_EDGE, TIM_LOOP);
  timer1_write(3125000);      // Arm the timer every 10s: 1 tick per 3.2us from TIM_DIV256 => 10s = 10.000.000us => 10000000/3.2 = 3125000 ticks
  
  while(1)
  {  
    co2Wert = ReadCO2(&temp);
    if (co2Wert != -1)
    {
      // compare CO2-value
      AlarmStatus = -1;
      digitalWrite(LEDred, LOW);     
      digitalWrite(LEDylw, LOW);     
      digitalWrite(LEDgrn, LOW);     
      if (co2Wert > LEVEL4)
      {
        digitalWrite(LEDred, HIGH);  
        PlayAlarm ();
        AlarmStatus = 1;
      }
      else if (co2Wert > LEVEL3)
      {
        digitalWrite(LEDylw, HIGH);  
        digitalWrite(LEDred, HIGH);  
      }
      else if (co2Wert > LEVEL2)
      {
        digitalWrite(LEDylw, HIGH);  
      }
      else if (co2Wert > LEVEL1)
      {
        digitalWrite(LEDylw, HIGH);  
        digitalWrite(LEDgrn, HIGH);     
      }
      else
        digitalWrite(LEDgrn, HIGH);     
        
#ifdef DEBUG 
      Serial.print("CO2: ");
      Serial.print(co2Wert);
      Serial.print(" ppm\t");
      Serial.print(temp);
      Serial.println(" *C");
#endif
      
#ifdef WLAN_SUPPORT
      if (cnt % 2 == 0)   // bei jedem 2. Durchlauf Daten senden
      {
        if (WiFi.status() == WL_CONNECTED)  // Wi-Fi connected? wenn nein: automatisch im Hintergrund permanentes reconnect
        {
          http.setReuse(true);
          APIurl = String (ThingSpeakURL + APIKey + "&field1=" + co2Wert + "&field2=" + temp);
          http.begin(APIurl);
          httpCode = http.GET();
          if (httpCode == HTTP_CODE_OK)    // 200 = OK
            FlashBlueLED ();
          http.end();
    
      #ifdef DEBUG 
          Serial.print("URL: "); 
          Serial.println(APIurl); 
          Serial.print("HTTP-Code: "); 
          Serial.println(httpCode); 
          
      #endif
    
          digitalWrite(LED_BUILTIN, LOW);     // blue LED on
        }
      }
#endif
    }
    cnt++;    // Zaehler mit Ueberlauf
    delay(tInterval * 1000);      //ms
  } // while (1)
}
 
